/** 
 (C) Copyright IBM Corporation 2006
 DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 sample code created by IBM Corporation. This sample code is not
 part of any standard or IBM product and is provided to you solely
 for  the purpose of assisting you in the development of your
 applications.  The code is provided "AS IS", without
 warranty of any kind.  IBM shall not be liable for any damages
 arising out of your use of the sample code, even if they have been
 advised of the possibility of such damages.
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.Dimension;
import java.awt.Toolkit;

import javax.swing.JFrame;

import jpos.JposConst;
import jpos.JposException;
import jpos.POSKeyboard;
import jpos.POSKeyboardConst;
import jpos.events.DataEvent;
import jpos.events.DataListener;
import jpos.events.StatusUpdateEvent;
import jpos.events.StatusUpdateListener;

/** 
 * POSKeyboardDemo class implements a demo 
 * for the basic functionality of the POSKeyboard
 * @author 
 * */

public class POSKeyboardDemo implements DeviceDemo,StatusUpdateListener,
                                        DataListener
{

    //-------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try POSKeyboardDemo  
     **/

    public static void main(String args[])
    {
        // main needs one logical name as argument 
        if (args.length != 1)
        {
            System.out.println("Usage : \n" +
            "java com.ibm.jpos.tools.sdicc.demo.POSKeyboardDemo <logicalName>");
            return;
        }
        //create this class that encapsulate the  jpos.POSKeyboard functionality
        DeviceDemo devDemo = new POSKeyboardDemo();
        //create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        //Add the ui to a JFrame
        JFrame mainWindow = new JFrame("POSKeyboardDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700,500);
        //Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (screenSize.width - mainWindow.getSize().width) / 2;
        int y = (screenSize.height - mainWindow.getSize().height) / 2;
        mainWindow.setLocation(x, y);
        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);
    }

    /**
     * @return a bidimensional arrays of strings
     *         the first array has the button labels
     *         the seccond array has the tooltips for those buttons 
     * */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_START_DEVICE },
        { TOOLTIP_TEXT_DEVICE_START } };

    }

    /** 
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "POSKeyboard";
    }

    /**
     * @param demoUI the ui to be used for this demo 
     * */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
    }

    /**
     * @param button the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_START_DEVICE))
        {
            // start the device test
            doStartDeviceTest();
            ui.flipButtonLabel(TEXT_START_DEVICE,
                    new String[]{TEXT_STOP_DEVICE,TOOLTIP_TEXT_DEVICE_STOP});
        }else if ( button.equals(TEXT_STOP_DEVICE ))
        {
            ui.flipButtonLabel(TEXT_STOP_DEVICE,
                    new String[]{TEXT_START_DEVICE,TOOLTIP_TEXT_DEVICE_START});
            doStopDeviceTest();
        }
    }
    
    /**
     * implementation of statusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param e the event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent e)
    {
        int status = e.getStatus();

        String statusMsg = "StatusUpdateOccurred = ";
        switch (status)
        {
            case JposConst.JPOS_SUE_POWER_ONLINE:
                statusMsg += "JPOS_SUE_POWER_ONLINE";
                break;

            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                statusMsg += "JPOS_SUE_POWER_OFF_OFFLINE";
                break;

        }
        //display the status update received
        ui.displayText(statusMsg);
    }

    public void dataOccurred(DataEvent e)
    {
        try{       
            StringBuffer sb = 
                new StringBuffer(dataCounter++ + ".-dataOccurred={");
            sb.append("POSKeyData Property = " );
            sb.append(" 0x" + Integer.toHexString( kbd.getPOSKeyData() ));
            sb.append(" (" + kbd.getPOSKeyData() +")" );
            sb.append(", POSKeyEventType Property = ");
            if ( kbd.getPOSKeyEventType() == POSKeyboardConst.KBD_KET_KEYDOWN)
                sb.append( "KBD_KET_KEYDOWN" );
            else
                sb.append( "KBD_KET_KEYUP" );
            sb.append("}");    
            ui.displayText(sb.toString());
            
            kbd.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
        }
        catch(JposException je)
        {
            String errMsg;
            errMsg = "JposException(" + je.getErrorCode() + ","
                    + je.getErrorCodeExtended() + "," + je.getMessage() + ")";
            ui.displayText(errMsg);
        }
    }



    //-------------------------------------------------------------------------
    // UPOS Functionality
    //

    //-------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * open claim and enable the device, display some properties
     * and open the POSKeyboard
     * */
    protected void doStartDeviceTest() throws JposException
    {
        kbd.open(ui.getLogicalName());
        ui.displayText("Open(\"" + ui.getLogicalName() + "\")");
        try{
            dataCounter = 0;
            // Common Cap/Properties
            ui.displayText("DeviceControlVersion = "
                    + kbd.getDeviceControlVersion());
            ui.displayText("DeviceControlDescription = "
                    + kbd.getDeviceControlDescription());
            ui.displayText("DeviceServiceVersion = "
                    + kbd.getDeviceServiceVersion());
            ui.displayText("DeviceServiceDescription = "
                    + kbd.getDeviceServiceDescription());
            ui.displayText("CapCompareFirmwareVersion = " + 
                    kbd.getCapCompareFirmwareVersion());
            String capPowerReportingName = "";
            switch(kbd.getCapPowerReporting())
            {
                case JposConst.JPOS_PR_NONE:
                    capPowerReportingName += "JPOS_PR_NONE = ";
                    break;
                case JposConst.JPOS_PR_STANDARD:
                    capPowerReportingName += "JPOS_PR_STANDARD = ";
                    break;
                case JposConst.JPOS_PR_ADVANCED:
                    capPowerReportingName += "JPOS_PR_ADVANCED = ";
                    break;
            }
            ui.displayText("CapPowerReporting = "+ capPowerReportingName + 
                    kbd.getCapPowerReporting());
            ui.displayText("CapStatisticsReporting = " + 
                    kbd.getCapStatisticsReporting());
            ui.displayText("CapUpdateFirmware = " +kbd.getCapUpdateFirmware());
            ui.displayText("CapUpdateStatistics = " + 
                    kbd.getCapUpdateStatistics());
            // Specific Cap/Properties
            ui.displayText("CapKeyUp = " + kbd.getCapKeyUp()); 
            kbd.claim(1000);
            ui.displayText("Claim(1000)");
            kbd.addStatusUpdateListener(this);
            ui.displayText("addStatusUpdateListener(this)");
            kbd.addDataListener(this);
            ui.displayText("addDataListener(this)");
            kbd.setEventTypes(POSKeyboardConst.KBD_ET_DOWN_UP);
            ui.displayText("setEventTypes(POSKeyboardConst.KBD_ET_DOWN_UP)");
            if( kbd.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
            {
                kbd.setPowerNotify(JposConst.JPOS_PN_ENABLED);
                ui.displayText("PowerNotify = JPOS_PN_ENABLED");
            }
            kbd.setDeviceEnabled(true);
            ui.displayText("setDeviceEnabled(true)");
            kbd.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
            ui.displayText("#\n# Press a key to test the POSKeyboard...\n#");
        }
        catch(JposException e)
        {
            doStopDeviceTest();
            throw e;
        }
    }

    /**
     * remove data listener
     * disable release and close the device
     * */
    protected void doStopDeviceTest() throws JposException
    {
        kbd.removeDataListener(this);
        ui.displayText("removeDataListener");
        kbd.removeStatusUpdateListener(this);
        ui.displayText("removeStatusUpdateListener(this)");
        kbd.close();
        ui.displayText("close()");
    }

    // -------------------------------------------------------------------------
    // instance variables
    //
    private POSKeyboard kbd = new POSKeyboard();
    private static DemoUI ui;
    private int dataCounter=0;

    //-------------------------------------------------------------------------
    // constants
    //
    /** Label for Start Button */
    private static final String TEXT_START_DEVICE="Start POSKeyboard Test";
    public static final String TEXT_STOP_DEVICE="Stop POSKeyboard Test";
    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_DEVICE_START = 
        "Open, claim and enable the POSKeyboard device and " +  
        "make the device ready to receive scan codes";
    /** tool tip text for Stop button */
    private static final String TOOLTIP_TEXT_DEVICE_STOP = 
        "Unregister the listener and closes the POSKeyboard";
}
