/** 
 (C) Copyright IBM Corporation 2006
 DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 sample code created by IBM Corporation. This sample code is not
 part of any standard or IBM product and is provided to you solely
 for  the purpose of assisting you in the development of your
 applications.  The code is provided "AS IS", without
 warranty of any kind.  IBM shall not be liable for any damages
 arising out of your use of the sample code, even if they have been
 advised of the possibility of such damages.
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.Dimension;
import java.awt.Toolkit;

import javax.swing.JFrame;
import javax.swing.JOptionPane;

import jpos.JposConst;
import jpos.JposException;
import jpos.HardTotals;
import jpos.events.StatusUpdateEvent;
import jpos.events.StatusUpdateListener;

/** 
 * HardTotalsDemo class implements a demo 
 * for the basic functionality of the HardTotals
 * @author 
 * */

public class HardTotalsDemo implements DeviceDemo, StatusUpdateListener
{

    //-------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try HardTotalsDemo  
     **/

    public static void main(String args[])
    {
        // main needs one logical name as argument 
        if (args.length != 1)
        {
            System.out.println("Usage : \n" +
            "java com.ibm.jpos.tools.sdicc.demo.HardTotalsDemo <logicalName>");
            return;
        }
        //create this class that encapsulate the  jpos.HardTotals functionality
        DeviceDemo devDemo = new HardTotalsDemo();
        //create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        //Add the ui to a JFrame
        JFrame mainWindow = new JFrame("HardTotalsDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700,400);
        //Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (screenSize.width - mainWindow.getSize().width) / 2;
        int y = (screenSize.height - mainWindow.getSize().height) / 2;
        mainWindow.setLocation(x, y);
        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);
    }

    /**
     * @return a bidimensional arrays of strings
     *         the first array has the button labels
     *         the seccond array has the tooltips for those buttons 
     * */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_HT_TEST },
        { TOOLTIP_TEXT_HT_TEST } };

    }

    /** 
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "HardTotals";
    }

    /**
     * @param demoUI the ui to be used for this demo 
     * */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
    }

    /**
     * @param button the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_HT_TEST))
        {
            // start the device test
            doStartDeviceTest();
        }

    }

    /**
     * implementation of statusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param e
     *            the event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent e)
    {
        int status = e.getStatus();

        String statusMsg = "StatusUpdateOccurred - ";
        switch (status)
        {
            case JposConst.JPOS_SUE_POWER_ONLINE:
                statusMsg += "JPOS_SUE_POWER_ONLINE";
                break;

            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                statusMsg += "JPOS_SUE_POWER_OFF_OFFLINE";
                break;

        }
        //display the status update received
        ui.displayText(statusMsg);

    }

    //-------------------------------------------------------------------------
    // UPOS Functionality
    //

    //-------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * open claim and enable the device, display some properties
     * and open the HardTotals
     * */
    protected void doStartDeviceTest() throws JposException
    {
        ht.open(ui.getLogicalName());
        ui.displayText("Open(\"" + ui.getLogicalName() + "\")");
        try{
            // Common Properties
            ui.displayText("DeviceControlVersion = "
                    + ht.getDeviceControlVersion());
            ui.displayText("DeviceControlDescription = "
                    + ht.getDeviceControlDescription());
            ui.displayText("DeviceServiceVersion = "
                    + ht.getDeviceServiceVersion());
            ui.displayText("DeviceServiceDescription = "
                    + ht.getDeviceServiceDescription());
            ui.displayText("CapCompareFirmwareVersion = " + 
                    ht.getCapCompareFirmwareVersion());
            String capPowerReportingName = "";
            switch(ht.getCapPowerReporting())
            {
                case JposConst.JPOS_PR_NONE:
                    capPowerReportingName += "JPOS_PR_NONE = ";
                    break;
                case JposConst.JPOS_PR_STANDARD:
                    capPowerReportingName += "JPOS_PR_STANDARD = ";
                    break;
                case JposConst.JPOS_PR_ADVANCED:
                    capPowerReportingName += "JPOS_PR_ADVANCED = ";
                    break;
            }
            ui.displayText("CapPowerReporting = "+ capPowerReportingName + 
                    ht.getCapPowerReporting());
            ui.displayText("CapStatisticsReporting = " + 
                    ht.getCapStatisticsReporting());
            ui.displayText("CapUpdateFirmware = " + ht.getCapUpdateFirmware());
            ui.displayText("CapUpdateStatistics = " + 
                    ht.getCapUpdateStatistics());
            ht.addStatusUpdateListener(this);
            ui.displayText("addStatusUpdateListener(this)");
            if( ht.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
            {
                ht.setPowerNotify(JposConst.JPOS_PN_ENABLED);
                ui.displayText("PowerNotify = JPOS_PN_ENABLED");
            }
            ht.setDeviceEnabled(true);
            ui.displayText("setDeviceEnabled(true)");
            ht.claim(1000);
            ui.displayText("Claim(1000)");

            // Specific Properties
            ui.displayText("getTotalsSize()=" + ht.getTotalsSize());
            ui.displayText("getFreeData()=" + ht.getFreeData());
            
            int[] handle = new int[1];
            ui.displayText("int[] handle = new int[1]");
            int[] size   = new int[1];
            ui.displayText("int[] size   = new int[1]");
            // verify if a HT file may exists
            try
            {
                ht.find( "",  handle, size );
                ui.displayText("find( \"\",  handle, size )");
                int reply = ui.displayConfirmDialog("File exists. Overwrite it?");
                switch (reply)
                {
                 case JOptionPane.OK_OPTION :
                     result = 1;
                     ht.delete("");
                     ui.displayText("delete(\"\")");
                     try
                        {
                            Thread.sleep(100);
                        }
                        catch (InterruptedException ie)
                        {
                        }
                     break;
                 case JOptionPane.CANCEL_OPTION :
                     result = 0;
                     break;
                 default :
                     result = 0;
                     break;
                }
            }
            catch(JposException e)
            {
                //Ignore file does not exists.
            }
            
            if(result != 0)
            {
                ht.create( "", handle, ht.getTotalsSize(), false );
                ui.displayText("create( \"\", handle, getTotalsSize(), false )");
                try
                {
                    Thread.sleep(100);
                }
                catch (InterruptedException ie)
                {
                }
                ht.claimFile(handle[0],2000);
                ui.displayText("claimFile(handle[0],2000)");
                
                String inputString = "WRITE THIS DATA INFO STRING TO HARD TOTALS";
                ui.displayText("String inputString=" +
                        "\"WRITE THIS DATA INFO STRING TO HARD TOTALS\"");

                byte[] inputArray = inputString.getBytes();
                ui.displayText("byte[] inputArray=inputString.getBytes()");
                
                ht.write( handle[0], inputArray, 0, inputArray.length );
                ui.displayText("write( handle[0], inputArray, 0, inputArray.length )");
                
                byte[] outputArray = new byte[ inputArray.length ];
                ui.displayText("byte[] outputArray = new byte[ inputArray.length ]");
                
                ht.read( handle[0], outputArray, 0, outputArray.length );
                ui.displayText("read( handle[0], outputArray, 0, outputArray.length )");
                
                if(java.util.Arrays.equals(inputArray, outputArray) == true)
                    ui.displayText("#\n#info written and info read is the same\n#");
                else
                    ui.displayText("#\n#info written and info read is different\n#");

                ht.delete("");
                ui.displayText("delete(\"\")");
            }
            else
            {
                ui.displayText("#\n#Demo canceled by user\n#");
            }
        }
        finally{
            ht.close();
            ui.displayText("close()");
        }
    }
        
    //-------------------------------------------------------------------------
    // instance variables
    //
    private HardTotals ht = new HardTotals();
    private static DemoUI ui;
    private int result = -1;
    
    //-------------------------------------------------------------------------
    // constants
    //
    /** Label for Start Button */
    private static final String TEXT_HT_TEST = "Start HardTotals Test";

    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_HT_TEST = 
        "Open, claim and enable the HardTotals. "
            + "find out if a previous HardTotals file exists. "
            + "Ask to \"delete it\" and continue or cancel "
            + "Create a new file. Write data to the file. "
            + "Read data sent to the file and compare "
            + "Delete the file created.";
}
