/** 
 (C) Copyright IBM Corporation 2006
 DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 sample code created by IBM Corporation. This sample code is not
 part of any standard or IBM product and is provided to you solely
 for  the purpose of assisting you in the development of your
 applications.  The code is provided "AS IS", without
 warranty of any kind.  IBM shall not be liable for any damages
 arising out of your use of the sample code, even if they have been
 advised of the possibility of such damages.
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.Dimension;
import java.awt.Toolkit;
import javax.swing.JFrame;
import jpos.JposConst;
import jpos.JposException;
import jpos.MSR;
import jpos.MSRConst;
import jpos.events.ErrorEvent;
import jpos.events.StatusUpdateEvent;
import jpos.events.StatusUpdateListener;
import jpos.events.DataEvent;
import jpos.events.DataListener;
import jpos.events.ErrorListener;

/**
 * MSRDemo class implements a demo for the basic functionality of the MSR
 * 
 * @author Lupita Morales
 */

public class MSRDemo implements DeviceDemo, StatusUpdateListener, DataListener,
        ErrorListener
{

    // -------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try MSRDemo
     */
    public static void main(String args[])
    {
        // main needs one logical name as argument
        if (args.length != 1)
        {
            System.out
                    .println("Usage : \n"
                            + "java com.ibm.jpos.tools.sdicc.demo.MSRDemo <logicalName>");
            return;
        }
        // create this class that encapsulate the jpos.MSR functionality
        DeviceDemo devDemo = new MSRDemo();
        // create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        // Add the ui to a JFrame
        mainWindow = new JFrame("MSRDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700, 500);
        // Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (screenSize.width - mainWindow.getSize().width) / 2;
        int y = (screenSize.height - mainWindow.getSize().height) / 2;
        mainWindow.setLocation(x, y);
        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);
    }

    /**
     * @return a bidimensional arrays of strings the first array has the button
     *         labels the seccond array has the tooltips for those buttons
     */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_MSR_TEST_START },
        { TOOLTIP_TEXT_MSR_START } };
    }

    /**
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "MSR";
    }

    /**
     * @param demoUI
     *            the ui to be used for this demo
     */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
    }

    /**
     * @param button
     *            the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_MSR_TEST_START))
        {
            // start the device test
            doStartDeviceTest();
            ui.flipButtonLabel(TEXT_MSR_TEST_START, new String[]
            { TEXT_MSR_TEST_STOP, TOOLTIP_TEXT_MSR_STOP });
        }
        else
        {
            doStopDeviceTest();
            ui.flipButtonLabel(TEXT_MSR_TEST_STOP, new String[]
            { TEXT_MSR_TEST_START, TOOLTIP_TEXT_MSR_START });
        }
    }

    /**
     * implementation of statusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param e
     *            the event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent e)
    {
        int status = e.getStatus();

        String statusMsg = "StatusUpdateOccurred = ";
        switch (status)
        {
            case JposConst.JPOS_SUE_POWER_ONLINE:
                statusMsg += "JPOS_SUE_POWER_ONLINE";
                break;

            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                statusMsg += "JPOS_SUE_POWER_OFF_OFFLINE";
                break;

        }
        // display the status update received
        ui.displayText(statusMsg);

    }

    /**
     * implementation of ErrorListener interface it is called when a error is
     * detected
     * 
     * @param e
     *            the error event generated by the device
     */
    public void errorOccurred(ErrorEvent e)
    {
        ui.displayText("errorEvent(" + e.getErrorCode() + ","
                + e.getErrorCodeExtended() + ")");
        try
        {
            msr.clearInput();
        }
        catch (JposException je)
        {
            je.printStackTrace();
        }
    }

    /**
     * implementation of DataListener interface it is called when a data event
     * is received from the MSR
     * 
     * @param e
     *            the event generated by the device
     */
    public void dataOccurred(DataEvent arg0)
    {
        ui.displayText("dataOccurred = ");
        displayData();
        try
        {
            ui.displayText("setDataEventEnabled(true)");
            msr.setDataEventEnabled(true);
        }
        catch(JposException je)
        {}
    }

    // -------------------------------------------------------------------------
    // UPOS Functionality
    //

    // -------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * open claim and enable the device, display some properties and open the
     * MSR
     */
    protected void doStartDeviceTest() throws JposException
    {
        msr.open(ui.getLogicalName());
        ui.displayText("Open()");
        ui.displayText("DeviceControlVersion = "
                + msr.getDeviceControlVersion());
        ui.displayText("DeviceControlDescription = "
                + msr.getDeviceControlDescription());
        ui.displayText("DeviceServiceVersion = "
                + msr.getDeviceServiceVersion());
        ui.displayText("DeviceServiceDescription = "
                + msr.getDeviceServiceDescription());
        ui.displayText("CapCompareFirmwareVersion = "
                + msr.getCapCompareFirmwareVersion());
        String capPowerReportingName = "";
        switch (msr.getCapPowerReporting())
        {
            case JposConst.JPOS_PR_NONE:
                capPowerReportingName += "JPOS_PR_NONE = ";
                break;
            case JposConst.JPOS_PR_STANDARD:
                capPowerReportingName += "JPOS_PR_STANDARD = ";
                break;
            case JposConst.JPOS_PR_ADVANCED:
                capPowerReportingName += "JPOS_PR_ADVANCED = ";
                break;
        }
        ui.displayText("CapPowerReporting = " + capPowerReportingName
                + msr.getCapPowerReporting());
        ui.displayText("CapStatisticsReporting = "
                + msr.getCapStatisticsReporting());
        ui.displayText("CapUpdateFirmware = " + msr.getCapUpdateFirmware());
        ui.displayText("CapUpdateStatistics = " + msr.getCapUpdateStatistics());
        // Specific Cap/Properties
        ui.displayText("CapJISOne = " + msr.getCapJISOne());
        ui.displayText("CapJISTwo = " + msr.getCapJISTwo());
        msr.claim(1000);
        ui.displayText("Claim(1000)");
        msr.addStatusUpdateListener(this);
        ui.displayText("addStatusUpdateListener(this)");
        msr.addDataListener(this);
        ui.displayText("addDataListener(this)");
        msr.addErrorListener(this);
        ui.displayText("addErrorListener(this)");
        if (msr.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
        {
            msr.setPowerNotify(JposConst.JPOS_PN_ENABLED);
            ui.displayText("PowerNotify = JPOS_PN_ENABLED");
        }
        msr.setDeviceEnabled(true);
        ui.displayText("setDeviceEnabled(true)");
        msr.setDataEventEnabled(true);
        ui.displayText("setDataEventEnabled(true)");
        ui.displayText("#\n# Please swipe a card \n#");
        if (msr.getCapJISTwo() == true)
        {
            // This is a DBCS MSR, so only Tracks 2, 4 should be set
            msr.setTracksToRead(MSRConst.MSR_TR_2_4);
            ui.displayText("setTracksToRead(MSR_TR_2_4)");
        }
        else
        {
            // This is an ISO MSR, all tracks can be read
            msr.setTracksToRead(MSRConst.MSR_TR_1_2_3_4);
            ui.displayText("setTracksToRead(MSR_TR_1_2_3_4)");
        }
    }

    /**
     * Removes the listeners and closes the device
     */
    protected void doStopDeviceTest() throws JposException
    {
        msr.removeStatusUpdateListener(this);
        ui.displayText("removeStatusUpdateListener(this)");
        msr.removeDataListener(this);
        ui.displayText("removeDataListener(this)");
        msr.release();
        ui.displayText("release()");
        msr.close();
        ui.displayText("close()");
    }

    /**
     * Converts the byte[] received to String
     */
    protected String decodeTrack(byte[] byteArray)
    {
        StringBuffer track = new StringBuffer();
        for (int i = 0; i < byteArray.length; ++i)
        {
            track.append((char) byteArray[i]);
        }
        return track.toString();
    }

    /**
     * Method called after a MSR data event was received. It will display every
     * track data
     * 
     */
    protected void displayData()
    {
        try
        {
            ui.displayText("Track1 property - "
                    + decodeTrack(msr.getTrack1Data()));
            ui.displayText("Track2 property - "
                    + decodeTrack(msr.getTrack2Data()));
            ui.displayText("Track3 property - "
                    + decodeTrack(msr.getTrack3Data()));
            ui.displayText("Track4 property - "
                    + decodeTrack(msr.getTrack4Data()));
        }
        catch (JposException je)
        {
            je.printStackTrace();
        }
    }

    // -------------------------------------------------------------------------
    // instance variables
    //
    private MSR msr = new MSR();

    private static DemoUI ui;

    private static JFrame mainWindow = null;

    // -------------------------------------------------------------------------
    // constants
    //
    /** Label for Start Button */
    private static final String TEXT_MSR_TEST_START = 
        "Start MSR Test";

    /** Label for Stop Button */
    private static final String TEXT_MSR_TEST_STOP = 
        "Stop MSR Test";

    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_MSR_START = 
        "Open, claim, enable the MSR "
            + "and prepare it for reading";

    /** tool tip text for Stop button */
    private static final String TOOLTIP_TEXT_MSR_STOP = 
        "Unregister the listener and closes the MSR";
}
