/** 
 (C) Copyright IBM Corporation 2006
 DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 sample code created by IBM Corporation. This sample code is not
 part of any standard or IBM product and is provided to you solely
 for  the purpose of assisting you in the development of your
 applications.  The code is provided "AS IS", without
 warranty of any kind.  IBM shall not be liable for any damages
 arising out of your use of the sample code, even if they have been
 advised of the possibility of such damages.
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.*;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

import com.ibm.jpos.services.DirectIO;

import jpos.JposConst;
import jpos.JposException;
import jpos.CheckScanner;
import jpos.CheckScannerConst;
import jpos.events.ErrorEvent;
import jpos.events.StatusUpdateEvent;
import jpos.events.ErrorListener;
import jpos.events.StatusUpdateListener;

/**
 * CheckScannerDemo class implements a demo for the basic functionality of the
 * CheckScanner
 * 
 * @author
 */

public class CheckScannerDemo implements DeviceDemo, StatusUpdateListener,
        ErrorListener
{

    // ------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try CheckScannerDemo
     */

    public static void main(String args[])
    {
        // main needs one logical name as argument
        if (args.length != 1)
        {
            System.out.println("Usage : \n  java " +
               "com.ibm.jpos.tools.sdicc.demo.CheckScannerDemo <logicalName>");
            return;
        }
        // create this class that encapsulate the jpos.CheckScanner
        // functionality
        DeviceDemo devDemo = new CheckScannerDemo();
        // create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        // Add the ui to a JFrame
        JFrame mainWindow = new JFrame("CheckScannerDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700, 500);
        // Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (screenSize.width - mainWindow.getSize().width) / 2;
        int y = (screenSize.height - mainWindow.getSize().height) / 2;
        mainWindow.setLocation(x, y);
        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);
    }

    /**
     * @return a bidimensional arrays of strings the first array has the button
     *         labels the seccond array has the tooltips for those buttons
     */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_CHKSCANNER_TEST_RTV_IMG, TEXT_CHKSCANNER_TEST_PRN_IMG },
        { TOOLTIP_TEXT_CHKSCANNER_RTV_IMG, TOOLTIP_TEXT_CHKSCANNER_PRN_IMG } };
    }

    /**
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "CheckScanner";
    }

    /**
     * @param demoUI
     *            the ui to be used for this demo
     */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
        ui.displayText("#\n#Please, insert a check in the slip and wait," +
        " Scan process may take several seconds\n#");
    }

    /**
     * @param button
     *            the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_CHKSCANNER_TEST_RTV_IMG))
        {
            // start the device test
            doRetrieveImageTest();
        }
        else if (button.equals(TEXT_CHKSCANNER_TEST_PRN_IMG))
        {
            // start the device test
            doPrintImageTest();
        }
    }

    /**
     * implementation of statusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param e
     *            the event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent e)
    {
        int status = e.getStatus();

        String statusMsg = "StatusUpdateOccurred = ";
        switch (status)
        {
            case JposConst.JPOS_SUE_POWER_ONLINE:
                statusMsg += "JPOS_SUE_POWER_ONLINE";
                break;

            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                statusMsg += "JPOS_SUE_POWER_OFF_OFFLINE";
                break;
            case CheckScannerConst.CHK_SUE_SCANCOMPLETE:
                statusMsg += "CHK_SUE_SCANCOMPLETE";
                synchronized (sulockObj)
                {
                    sulockObj.notifyAll();
                }
                break;
        }
        // display the status update received
        ui.displayText(statusMsg);
    }

    
    public void errorOccurred(ErrorEvent ee)
    {
        int ec  = ee.getErrorCode();
        int ece = ee.getErrorCodeExtended();
        String statusMsg = "\n#\n# ErrorOccurred: " +
                DemoUI.getErrorCodeName(ec) + ((ec==JposConst.JPOS_E_EXTENDED)?
                        ":" + DemoUI.getErrorCodeName(ece):"") + "\n#\n";
        ui.displayText(statusMsg);
    }

    // -------------------------------------------------------------------------
    // UPOS Functionality
    //

    // -------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * Method to display basic properties of the CheckScanner
     */
    protected void displayProperties() throws JposException
    {
        // Common Properties
        ui.displayText("DeviceControlVersion = "
                + ckScanner.getDeviceControlVersion());
        ui.displayText("DeviceControlDescription = "
                + ckScanner.getDeviceControlDescription());
        ui.displayText("DeviceServiceVersion = "
                + ckScanner.getDeviceServiceVersion());
        ui.displayText("DeviceServiceDescription = "
                + ckScanner.getDeviceServiceDescription());
        ui.displayText("CapCompareFirmwareVersion = "
                + ckScanner.getCapCompareFirmwareVersion());
        String capPowerReportingName = "";
        switch (ckScanner.getCapPowerReporting())
        {
            case JposConst.JPOS_PR_NONE:
                capPowerReportingName += "JPOS_PR_NONE = ";
                break;
            case JposConst.JPOS_PR_STANDARD:
                capPowerReportingName += "JPOS_PR_STANDARD = ";
                break;
            case JposConst.JPOS_PR_ADVANCED:
                capPowerReportingName += "JPOS_PR_ADVANCED = ";
                break;
        }
        ui.displayText("CapPowerReporting = " + capPowerReportingName
                + ckScanner.getCapPowerReporting());
        ui.displayText("CapStatisticsReporting = "
                + ckScanner.getCapStatisticsReporting());
        ui.displayText("CapUpdateFirmware = "
                + ckScanner.getCapUpdateFirmware());
        ui.displayText("CapUpdateStatistics = "
                + ckScanner.getCapUpdateStatistics());
        ckScanner.addStatusUpdateListener(this);
        ui.displayText("addStatusUpdateListener(this)");
        if (ckScanner.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
        {
            ckScanner.setPowerNotify(JposConst.JPOS_PN_ENABLED);
            ui.displayText("PowerNotify = JPOS_PN_ENABLED");
        }
    }
    /**
     * open claim and enable the device, display some properties and calls the
     * retrieveImage method, then displays the image in a JPaneOption
     */
    protected void doRetrieveImageTest() throws JposException
    {
        ui.displayText("=======================================================");
        ckScanner.open(ui.getLogicalName());
        ui.displayText("Open(\"" + ui.getLogicalName() + "\")");
        try
        {
            displayProperties();
            ckScanner.claim(1000);
            ui.displayText("Claim(1000)");
            ckScanner.setDeviceEnabled(true);
            ui.displayText("setDeviceEnabled(true)");
            ckScanner.setConcurrentMICR(false);
            ui.displayText("setConcurrentMICR(false)");
            ckScanner.clearImage(CheckScannerConst.CHK_CLR_ALL);
            ui.displayText("clearImage = CHK_CLR_ALL");
            ckScanner.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
            ckScanner.setImageFormat(CheckScannerConst.CHK_IF_JPEG);
            ui.displayText("setImageFormat(CheckScannerConst.CHK_IF_JPEG)");
            ui.displayText("#\n#Please Insert a check timeout 5000\n#");
            ckScanner.beginInsertion(TIMEOUT);
            ui.displayText("beginInsertion(" + TIMEOUT + ")");
            synchronized (sulockObj)
            {
                ckScanner.endInsertion();
                ui.displayText("endInsertion()");
                waitStatusEvent();
                createImageFrame();
            }
        }
        finally
        {
            ckScanner.removeStatusUpdateListener(this);
            ui.displayText("removeStatusUpdateListener(this)");
            ckScanner.close();
            ui.displayText("close()");
        }
    }
    
    /**
     * open claim and enable the device, display some properties and print the
     * scanned image
     */
    protected void doPrintImageTest() throws JposException
    {
        int data[] = new int[5];
        data[0] = DirectIO.CHK_DIO_MEMORY_IMG;
        data[1] = jpos.CheckScannerConst.CHK_CROP_AREA_ENTIRE_IMAGE;
        data[2] = 1;
        data[3] = 1;
        data[4] = DirectIO.CHK_DIO_NO_ROTATE;
        
        ckScanner.open(ui.getLogicalName());
        ui.displayText("Open(\"" + ui.getLogicalName() + "\")");
        try
        {
            displayProperties();
            ckScanner.claim(1000);
            ui.displayText("Claim(1000)");
            ckScanner.setDeviceEnabled(true);
            ui.displayText("setDeviceEnabled(true)");
            ckScanner.setConcurrentMICR(false);
            ui.displayText("setConcurrentMICR(false)");
            ckScanner.clearImage(CheckScannerConst.CHK_CLR_ALL);
            ui.displayText("clearImage = CHK_CLR_ALL");
            ckScanner.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
            ui.displayText("#\n#Please Insert a check timeout 5000\n#");
            ckScanner.beginInsertion(TIMEOUT);
            ui.displayText("beginInsertion(" + TIMEOUT + ")");
            synchronized (sulockObj)
            {
                ckScanner.endInsertion();
                ui.displayText("endInsertion()");
                waitStatusEvent();
                ckScanner.directIO(DirectIO.CHK_DIO_PRINT_SCANNED_IMAGE_CMD, data, null);
            }
        }
        finally
        {
            ckScanner.removeStatusUpdateListener(this);
            ui.displayText("removeStatusUpdateListener(this)");
            ckScanner.close();
            ui.displayText("close()");
        }
    }

    /**
     * Create a command to retrieve the property imageData and display the
     * scanned image,it will also send the image to the printer receipt station
     * 
     */
    private void createImageFrame() throws JposException
    {
        byte[] imageData = null;

        ui.displayText("retrieveImage(CHK_CROP_AREA_ENTIRE_IMAGE)");
        ckScanner
              .retrieveImage(CheckScannerConst.CHK_CROP_AREA_ENTIRE_IMAGE);
        imageData = ckScanner.getImageData();
        ui.displayText("getImageData size: " + imageData.length);
        Image image = null;
        image = Toolkit.getDefaultToolkit().createImage(imageData, 0,
                imageData.length);
        ui.displayText("#\n# Scaling image for display... \n#");
        Image scaledImage = image.getScaledInstance(DEFAULT_WIDTH,
                DEFAULT_HEIGHT, Image.SCALE_FAST);
        ui.displayText("# Scaled from : " + ckScanner.getDocumentWidth() + "x" +
                ckScanner.getDocumentHeight() + " to " + DEFAULT_WIDTH +
                "x" + DEFAULT_HEIGHT + "\n#");
        
        // Now it is all set to display the image
        Object[] options = {"OK"};
        JOptionPane.showOptionDialog(ui, "", "Scanned Check", 
        JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE,
        new ImageIcon(scaledImage), options, options[0]);
    }

    /**
     * Method to lock the device and wait for a status update event that
     * notifies that a new scanned image has arrived
     * 
     */
    protected synchronized void waitStatusEvent()
    {
        synchronized (sulockObj)
        {
            try
            {
                sulockObj.wait(5000);
            }
            catch (InterruptedException ie)
            {
            }
        }
    }

    // ------------------------------------------------------------------------
    // instance variables
    //
    private CheckScanner ckScanner = new CheckScanner();

    private static DemoUI ui;

    private Object sulockObj = new Object();

    // ------------------------------------------------------------------------
    // constants
    //

    /** Label for Retrieve Image Button */
    private static final String TEXT_CHKSCANNER_TEST_RTV_IMG = 
        "RetrieveImage Test";

    /** Label for the start Print button */
    private static final String TEXT_CHKSCANNER_TEST_PRN_IMG = 
        "Print check Test";
    
    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_CHKSCANNER_RTV_IMG = 
        "Open, claim, enable the CheckScanner "
         + "and retrieve the scanned image";
    
    /** Tool tip text for the Print Button */
    private static final String TOOLTIP_TEXT_CHKSCANNER_PRN_IMG = 
        "Open, claim, enable the CheckScanner "
         + "and retrieve the scanned image";

    /** Timeout for the beginInsertion method : 20 seconds */
    private static final int TIMEOUT = 5000;
    
    /** Constant for width after scalling */
    private static final int DEFAULT_WIDTH = 200;
    
    /** Constant for width after scalling */
    private static final int DEFAULT_HEIGHT = 400;

}
