/** 
 (C) Copyright IBM Corporation 2006
 DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
 sample code created by IBM Corporation. This sample code is not
 part of any standard or IBM product and is provided to you solely
 for  the purpose of assisting you in the development of your
 applications.  The code is provided "AS IS", without
 warranty of any kind.  IBM shall not be liable for any damages
 arising out of your use of the sample code, even if they have been
 advised of the possibility of such damages.
 */

package com.ibm.jpos.tools.sdicc.demo;

import java.awt.Dimension;
import java.awt.Toolkit;

import javax.swing.JFrame;

import jpos.JposConst;
import jpos.JposException;
import jpos.MICR;
import jpos.MICRConst;
import jpos.events.DataEvent;
import jpos.events.DataListener;
import jpos.events.ErrorEvent;
import jpos.events.ErrorListener;
import jpos.events.StatusUpdateEvent;
import jpos.events.StatusUpdateListener;

/** 
 * MICRDemo class implements a demo 
 * for the basic functionality of the MICR
 * @author 
 * */

public class MICRDemo implements DeviceDemo, StatusUpdateListener, DataListener, ErrorListener
{

    //-------------------------------------------------------------------------
    // public methods
    //

    /**
     * main method creates a GUI to try MICRDemo  
     **/

    public static void main(String args[])
    {
        // main needs one logical name as argument 
        if (args.length != 1)
        {
            System.out.println("Usage : \n" +
            "java com.ibm.jpos.tools.sdicc.demo.MICRDemo <logicalName>");
            return;
        }
        //create this class that encapsulate the  jpos.MICR functionality
        DeviceDemo devDemo = new MICRDemo();
        //create the UI for this demo
        DemoUI demoUI = new DemoUI(args, devDemo);
        //Add the ui to a JFrame
        JFrame mainWindow = new JFrame("MICRDemo");
        mainWindow.setContentPane(demoUI);
        mainWindow.setSize(700,400);
        //Center the Frame
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (screenSize.width - mainWindow.getSize().width) / 2;
        int y = (screenSize.height - mainWindow.getSize().height) / 2;
        mainWindow.setLocation(x, y);
        mainWindow.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        mainWindow.setVisible(true);

    }

    /**
     * @return a bidimensional arrays of strings
     *         the first array has the button labels
     *         the seccond array has the tooltips for those buttons 
     * */
    public String[][] getButtonLabels()
    {
        return new String[][]
        {
        { TEXT_MICR_TEST },
        { TOOLTIP_TEXT_MICR_TEST } };

    }

    /** 
     * @return the devCat for this device
     */
    public String getDevCat()
    {
        return "MICR";
    }

    /**
     * @param demoUI the ui to be used for this demo 
     * */
    public void setUI(DemoUI demoUI)
    {
        ui = demoUI;
        ui.displayText("#\n#Please, insert a check into the slip, then click" +
                " on the \"Start MICR Test\" button\n#");
    }

    /**
     * @param button the button that was pressed
     */
    public void buttonPressed(String button) throws JposException
    {
        if (button.equals(TEXT_MICR_TEST))
        {
            // start the device test
            doStartDeviceTest();
        }

    }

    /**
     * Implementation of StatusUpdateListener interface it is called when the
     * status of the device is updated
     * 
     * @param sue
     *            The Status Update Event generated by the device
     */
    public void statusUpdateOccurred(StatusUpdateEvent sue)
    {
        String statusMsg = "StatusUpdateOccurred() - " +
                translateStatusUpdateEvent(sue.getStatus());
        ui.displayText(statusMsg);
    }

    /**
     * Implementation of DataListener interface it is called when input data
     * from the device is ready.
     * 
     * @param de
     *            The Data Event generated by the device
     */
    public void dataOccurred(DataEvent de)
    {
        try
        {
            String statusMsg = "dataOccurred={\n" +
            "        RawData Property = \"" + micr.getRawData() + "\"\n" +
            "        TransitNumber Property = \"" + micr.getTransitNumber() + "\"\n" +
            "        AccountNumber Property = \"" + micr.getAccountNumber() + "\"\n" +
            "        SerialNumber Property = \"" + micr.getSerialNumber() + "\"\n" +
            "        BankNumber Property = \"" + micr.getBankNumber() + "\"\n" +
            "        CheckType Property = \"" + micr.getCheckType() + "\"\n" +
            "        CountryCode Property = \"" + micr.getCountryCode() + "\"\n" +
            "        Amount Property = \"" + micr.getAmount() + "\"\n" +
            "        EPC Property = \"" + micr.getEPC() + "\"}";
            ui.displayText(statusMsg);
            micr.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
        }
        catch (JposException e)
        { }
    }

    /**
     * Implementation of ErrorListener interface it is called when an error
     * ocurred in the device
     * 
     * @param ee
     *            The Error Event generated by the device
     */
    public void errorOccurred(ErrorEvent ee)
    {
        int ec  = ee.getErrorCode();
        int ece = ee.getErrorCodeExtended();
        String statusMsg = "\n#\n# ErrorOccurred: " +
                DemoUI.getErrorCodeName(ec) + ((ec==JposConst.JPOS_E_EXTENDED)?
                        ":" + translateErrorExtendedEvent(ece):"") + "\n#\n";
        ui.displayText(statusMsg);
    }

    //-------------------------------------------------------------------------
    // UPOS Functionality
    //

    //-------------------------------------------------------------------------
    // Protected Methods
    //

    /**
     * open claim and enable the device, display some properties
     * and open the MICR
     * */
    protected void doStartDeviceTest() throws JposException
    {
        ui.displayText("====================================================" +
                "=========================");
        micr.open(ui.getLogicalName());
        ui.displayText("open(\"" + ui.getLogicalName() + "\")");
        try{
            // Common Properties
            ui.displayText("DeviceControlVersion = "
                    + micr.getDeviceControlVersion());
            ui.displayText("DeviceControlDescription = "
                    + micr.getDeviceControlDescription());
            ui.displayText("DeviceServiceVersion = "
                    + micr.getDeviceServiceVersion());
            ui.displayText("DeviceServiceDescription = "
                    + micr.getDeviceServiceDescription());
            ui.displayText("CapCompareFirmwareVersion = " + 
                    micr.getCapCompareFirmwareVersion());
            String capPowerReportingName = "";
            switch(micr.getCapPowerReporting())
            {
                case JposConst.JPOS_PR_NONE:
                    capPowerReportingName += "JPOS_PR_NONE = ";
                    break;
                case JposConst.JPOS_PR_STANDARD:
                    capPowerReportingName += "JPOS_PR_STANDARD = ";
                    break;
                case JposConst.JPOS_PR_ADVANCED:
                    capPowerReportingName += "JPOS_PR_ADVANCED = ";
                    break;
            }
            ui.displayText("CapPowerReporting = "+ capPowerReportingName +
                    micr.getCapPowerReporting());
            ui.displayText("CapStatisticsReporting = " + 
                    micr.getCapStatisticsReporting());
            ui.displayText("CapUpdateFirmware = " +
                    micr.getCapUpdateFirmware());
            ui.displayText("CapUpdateStatistics = " + 
                    micr.getCapUpdateStatistics());
            ui.displayText("CapValidationDevice = " + 
                    micr.getCapValidationDevice());
            micr.setDataEventEnabled(true);
            ui.displayText("setDataEventEnabled(true)");
            micr.addStatusUpdateListener(this);
            ui.displayText("addStatusUpdateListener(this)");
            micr.addDataListener(this);
            ui.displayText("addDataListener(this)");
            micr.addErrorListener(this);
            ui.displayText("addErrorListener(this)");
            if( micr.getCapPowerReporting() != JposConst.JPOS_PR_NONE)
            {
                micr.setPowerNotify(JposConst.JPOS_PN_ENABLED);
                ui.displayText("PowerNotify = JPOS_PN_ENABLED");
            }
            micr.claim(1000);
            ui.displayText("claim(1000)");
            micr.setDeviceEnabled(true);
            ui.displayText("setDeviceEnabled(true)");
            // Specific Properties
            ui.displayText("#\n#Please Insert a check timeout 5000\n#");
            micr.beginInsertion(5000);
            ui.displayText("beginInsertion(5000)");
            micr.endInsertion();
            ui.displayText("endInsertion()");
            micr.beginRemoval(1000);
            ui.displayText("beginRemoval(1000)");
            micr.endRemoval();
            ui.displayText("endRemoval()");
        }
        finally
        {
            synchronized (this)
            {
                try
                {
                    this.wait(1000);
                }
                catch (InterruptedException e)
                { }
            }
            micr.removeStatusUpdateListener(this);
            ui.displayText("removeStatusUpdateListener(this)");
            micr.removeDataListener(this);
            ui.displayText("removeDataListener(this)");
            micr.removeErrorListener(this);
            ui.displayText("removeErrorListener(this)");
            micr.close();
            ui.displayText("close()");
        }
    }

    protected String translateStatusUpdateEvent(int sue)
    {
        String ret;
        switch (sue) {
            case JposConst.JPOS_SUE_POWER_OFF:
                ret = "JPOS_SUE_POWER_OFF";
                break;
            case JposConst.JPOS_SUE_POWER_OFF_OFFLINE:
                ret = "JPOS_SUE_POWER_OFF_OFFLINE";
                break;
            case JposConst.JPOS_SUE_POWER_OFFLINE:
                ret = "JPOS_SUE_POWER_OFFLINE";
                break;
            case JposConst.JPOS_SUE_POWER_ONLINE:
                ret = "JPOS_SUE_POWER_ONLINE";
                break;
                
            default:
                ret = "<unknown>";
                break;
        }
        return ret;
    }

    protected String translateErrorExtendedEvent(int sue)
    {
        String ret;
        switch (sue) {
            case MICRConst.JPOS_EMICR_CHECK:
                ret = "JPOS_EMICR_CHECK";
                break;
            case MICRConst.JPOS_EMICR_NOCHECK:
                ret = "JPOS_EMICR_NOCHECK";
                break;
            case MICRConst.JPOS_EMICR_BADDATA:
                ret = "JPOS_EMICR_BADDATA";
                break;
            case MICRConst.JPOS_EMICR_BADSIZE:
                ret = "JPOS_EMICR_BADSIZE";
                break;
            case MICRConst.JPOS_EMICR_CHECKDIGIT:
                ret = "JPOS_EMICR_CHECKDIGIT";
                break;
            case MICRConst.JPOS_EMICR_COVEROPEN:
                ret = "JPOS_EMICR_COVEROPEN";
                break;
            case MICRConst.JPOS_EMICR_JAM:
                ret = "JPOS_EMICR_JAM";
                break;
            case MICRConst.JPOS_EMICR_NODATA:
                ret = "JPOS_EMICR_NODATA";
                break;

            default:
                ret = "<unknown>";
                break;
        }
        return ret;
    }

    //-------------------------------------------------------------------------
    // instance variables
    //
    private MICR micr = new MICR();
    private static DemoUI ui;

    //-------------------------------------------------------------------------
    // constants
    //
    /** Label for Start Button */
    private static final String TEXT_MICR_TEST = "Start MICR Test";

    /** tool tip text for Start button */
    private static final String TOOLTIP_TEXT_MICR_TEST = 
        "Open, claim and enable the MICR, check for properties, process the " +
        "check and print Properies extracted from the check";


}
